#!/usr/bin/env tclsh
# chime.tcl - Copyright (C) 2005 Pat Thoyts <patthoyts@users.sourceforge.net>
#
# Chime component. Connects to a chat room and issues a time message on
# the hour each hour.
#
# -------------------------------------------------------------------------
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
# -------------------------------------------------------------------------

source [file join [file dirname [info script]] wrapper.tcl]
source [file join [file dirname [info script]] jcp.tcl]

package require xmppd::jcp;             # tclxmppd
package require wrapper;                # jabberlib

namespace eval ::chime {
    variable version 1.0.0
    variable rcsid {$Id: chime.tcl,v 1.2 2005/01/15 23:43:59 pat Exp $}

    variable Options
    if {![info exists Options]} {
        array set Options {
            JID            {}
            Name           Chime
            Resource       chime
            Conference     {}
    
	    JabberServer   {}
            JabberPort     5347
            Secret         {}

            LogLevel       notice
            LogFile        {}
        }
    }
    
    variable Component
}

# chime::start --
#
#	Start the chime component. This uses the jabber component protocol
#	to connect to the server and schedules the chimes.
#
proc ::chime::start {} {
    variable Options
    variable Component
    xmppd::jcp::configure \
        -component $Options(JID) \
        -secret    $Options(Secret) \
        -loglevel  $Options(LogLevel) \
        -handler   [namespace current]::Handler
    set Component [xmppd::jcp::create \
                       $Options(JabberServer) $Options(JabberPort)]
    after 200 [list [namespace origin presence] \
                   $Options(Name) available online {Hourly chime}]
    chimes start
    return
}

# chime::stop --
#
#	Halt the chime component.
#
proc ::chime::stop {} {
    variable Options
    variable Component
    chimes stop
    presence $Options(Name) unavailable
    xmppd::jcp::destroy $Component
}

# chime::Handler --
#
#	Jabber message routing. For this component, we don't need to
#	do anything as all we do is issue a time message on the hour.
#
proc ::chime::Handler {type attributes close value children} {
    switch -exact -- $type {
        message {}
        presence {}
        iq {}
        default {}
    }
    return
}

# chime::presence --
#
#	Send a jabber presence message
#
proc ::chime::presence {nick type {show {online}} {status {}} {user {}}} {
    variable Options
    variable Component

    if {$user eq {}} {set user $nick}
    set kids {} ; set hist {}
    set ts [clock format [clock seconds] -format %Y%m%dT%H:%M:%S -gmt 1]
    lappend hist [list history [list maxchars 0 maxstanzas 0] 1 "" {}]
    lappend kids [list x {xmlns http://jabber.org/protocols/muc} 0 "" $hist]
    if {$show ne {}} {
        lappend kids [list show {} 0 $show {}]
    }
    if {$status ne {}} {
        lappend kids [list status {
            xmlns:xml http://www.w3.org/XML/1998/namespace
            xml:lang en-GB
        } 0 $status {}]
    }
    set to $Options(Conference)
    if {$nick ne ""} {append to "/$nick"}
    set from ${user}@$Options(JID)/$Options(Resource)
    set attr [list from $from to $to xmlns jabber:client]
    if {$type ne {}} {lappend attr type $type}
    
    xmppd::jcp::route $Component \
        [wrapper::createxml [list presence $attr 0 "" $kids]]
    return
}

# chime::LoadConfig --
#
#	This procedure reads a text file and updates the Options array
#	from the contents. Comments and blank lines are ignored. All 
#	other lines must be a list of two elements, the first element 
#	must be an item in the Options array.
#
proc ::chime::LoadConfig {{conf {}}} {
    variable Options
    if {$conf eq {}} {
        set conf [file normalize [info script]]
        set base [file rootname [file tail $conf]].conf
        set conf [file join [file dirname $conf] $base]
    }
    if {[file exists $conf]} {
        set f [open $conf r]
        set n 0
        while {![eof $f]} {
            gets $f line
            string trim $line
            if {[string match "#*" $line]} continue
            if {[string length $line] < 1} continue
            if {[llength $line] != 2} {
                return -code error "invalid config line $n: \"$line\""
            }
            if {![info exists Options([lindex $line 0])]} {
                return -code error "invalid config option\
                \"[lindex $line 0]\" at line $n"
            }
            set Options([lindex $line 0]) [lindex $line 1]
            incr n
        }
        close $f
    } else {
        log warn "configuration file \"$conf\" could not be opened"
    }
    return
}

# chime::chimes --
#
#	Manage the scheduling of chimes on the hour.
#
proc ::chime::chimes {cmd} {
    variable ChimeId
    switch -exact -- $cmd {
        start {
            set ChimeId [after [nextchime] [namespace origin bong]]
        }
        stop {
            after cancel $ChimeId
        }
        default {
            return -code error "invalid option \"$cmd\": rtfm"
        }
    }
}

# chime::nextchime --
#
#	Calculate the number of milliseconds until the next hour.
#
proc ::chime::nextchime {} {
    set t [clock format [clock scan "+1 hour"] -format "%Y%m%d %H:00:00"]
    set delta [expr {([clock scan $t] - [clock seconds]) * 1000}]
    if {$delta < 60000} {
        puts stderr "error: chiming too fast"
        set delta 60000
    }
    puts "Schedule chime in $delta milliseconds"
    return $delta
}

# chime::bong --
#
#	Issue a timestamp message to the connected chatroom.
#
proc ::chime::bong {} {
    variable ChimeId
    variable Options
    variable Component

    after cancel $ChimeId
    set kids {}
    set ts [clock format [clock seconds] -format %Y%m%dT%H:%M:%S -gmt 1]
    puts "BONG at $ts"
    lappend kids [list body {} 0 \
                      [clock format [clock seconds] -gmt 1] {}]
    set from "$Options(Name)@$Options(JID)/$Options(Resource)"
    set attr [list from $from to $Options(Conference) \
                  type groupchat xmlns "jabber:client"]
    set xml [wrapper::createxml [list message $attr 0 "" $kids]]

    xmppd::jcp::route $Component $xml
    set ChimeId [after [nextchime] [namespace origin bong]]
}

chime::LoadConfig

if {!$tcl_interactive} {
    if {$tcl_platform(platform) eq "unix"} {
        set cmdloop [file join [file dirname [info script]] cmdloop.tcl]
        if {[file exist $cmdloop]} {
            source $cmdloop
            set cmdloop::welcome "Jabber chime bot $::chime::version"
            append cmdloop::welcome "\nReady on %client %port"
            cmdloop::cmdloop
            #cmdloop::listen 127.0.0.1 5442
        }
        set tcl_interactive 1;# don't do this again if we re-source.
    }

    chime::start

    if {![info exists tcl_service]} {
        vwait ::forever
        chime::stop
    }
}
