#!/usr/bin/env tclsh
# ijbridge.tcl - Copyright (C) 2004 Pat Thoyts <patthoyts@users.sf.net>
#
# This implements a bridge between a Jabber Multi-User Chat and an IRC
# channel.
#
# The IRC portions of this have come from ircbridge.
#
# -------------------------------------------------------------------------
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
# -------------------------------------------------------------------------

source [file join [file dirname [info script]] wrapper.tcl]
source [file join [file dirname [info script]] jcp.tcl]

package require xmppd::jcp;             # tclxmppd
package require wrapper;                # jabberlib
package require sha1;                   # tcllib
package require logger;                 # tcllib
package require irc 0.4;                # tcllib

namespace eval ::client {}

namespace eval ::ijbridge {

    variable version 2.0.0
    variable rcsid {$Id: ijbridge.tcl,v 1.10 2005/01/15 23:47:25 pat Exp $}

    # This array MUST be set up by reading the configuration file. The
    # member names given here define the settings permitted in the 
    # config file.
    # This script will not work by default - you MUST provide suitable 
    # connection details.
    #
    variable Options
    if {![info exists Options]} {
        array set Options {
            JID            {}
            Name           Bridge
            Resource       ijbridge
            Conference     {}
    
	    JabberServer   {}
            JabberPort     5347
            Secret         {}

            IrcServer      irc.freenode.net
            IrcPort        6667
            IrcUser        {}
            IrcChannel     {}

            LogLevel       notice
            LogFile        {}
            Chimes         no
        }
    }

    variable Component

    # State variable used in rate-limiting communications with IRC.
    variable Limit
    if {![info exists Limit]} {
        array set Limit {}
    }

    variable lastmsg
    if {![info exists lastmsg]} { array set lastmsg {} }

    variable mid
    if {![info exists mid]} { set mid 0 }
    proc mid {} {variable mid; return [set mid [expr {wide($mid) + 1}]]}

    # Used to maintain a view of the users currently connected to IRC.
    variable IrcUserList
    if {![info exists IrcUserList]} {
        set IrcUserList [list]
    }

    variable log
    if {![info exists log]} {
        set log [logger::init ijbridge]
        ${log}::setlevel $Options(LogLevel)
        namespace eval $log {variable logfile {}}
        if {$Options(LogFile) ne ""} {
            set ${log}::logfile [open $Options(LogFile) a+]
            namespace eval $log {
                fconfigure $logfile -buffering line
                puts $logfile [string repeat - 72]
                puts $logfile [clock format [clock seconds]]
            }
        }
        proc ${log}::stdoutcmd {level text} {
            variable service
            variable logfile
            set ts [clock format [clock seconds] -format {%H:%M:%S}]
            if {$logfile ne ""} {
                puts $logfile "\[$ts\] $level $text"
            }
            puts stdout $text
        }
        proc log {level text} {
            variable log
            ${log}::$level $text
        }
    }
}

# -------------------------------------------------------------------------

# ijbridge::start --
#
#	Starts the server. The Options array must have been properly 
#	set up (usually by calling ijbridge::LoadCOnfig) if this is
#	to work properly.
#
proc ::ijbridge::start {} {
    variable Options
    variable Component

    xmppd::jcp::configure \
        -component $Options(JID) \
        -secret    $Options(Secret) \
        -loglevel  $Options(LogLevel) \
        -handler   [namespace current]::OnJabber
    set Component [xmppd::jcp::create \
                       $Options(JabberServer) $Options(JabberPort)]

    after 200 [list [namespace origin presence] \
	$Options(Name) available online {Jabber to IRC channel bridge}]

    if {$Options(Chimes)} { chimes start }
    return
}

# ijbridge::stop --
#
#	Removes all the IRC users from the chat room and then closes all
#	the components.
#
proc ::ijbridge::stop {} {
    variable Options
    variable IrcUserList
    variable Component

    foreach nick $IrcUserList {
        presence $nick unavailable        
    }
    presence $Options(Name) unavailable
    chimes stop
    client::stop
    xmppd::jcp::destroy $Component
}

# ijbridge::loglevel --
#
#	Configure the logging level for all the components involved in
#	this bridge.
#	One of debug, info, notice, warn, emerg, crit
#
proc ::ijbridge::loglevel {level} {
    ${::ijbridge::log}::setlevel $level
    ${::client::log}::setlevel $level
    xmppd::jcp::configure -loglevel $level
    return $level
}

# ijbridge::OnJabber --
#
#	Called after the initial parsing of jabber messages by the
#	s2s component. This is the primary dispather for all jabber
#	messages.
#
proc ::ijbridge::OnJabber {type attributes close value children} {
    switch -exact -- $type {
        message {
            set msg {}
            foreach {a v} $attributes {
                if {$a eq "type"} {
                    set type $v
                } else {
                    lappend msg -$a $v
                }
            }
            foreach child $children {
                set elt [wrapper::gettag $child]
                if {$elt eq "body" || $elt eq "subject"} {
                    lappend msg -$elt [wrapper::getcdata $child]
                }
            }
            lappend msg -x $children
            if {[catch { eval [linsert $msg 0 OnMessage $type] } errmsg]} {
                log error $errmsg
            }
        }
        presence {
            set type available
            array set opts {-show online -status {} -priority 0}
            foreach {a v} $attributes {
                if {$a eq "type"} {
                    set type $v
                } else {
                    set opts(-$a) $v
                }
            }
            foreach child $children {
                switch -exact -- [wrapper::gettag $child] {
                    show   { set opts(-show) [wrapper::getcdata $child] }
                    status { set opts(-status) [wrapper::getcdata $child] }
                    priority { set opts(-priority) [wrapper::getcdata $child] }
                }
            }
            set opts(-x) $children
            if {[catch {
                eval [linsert [array get opts] 0 OnPresence $type]
            } msg]} {
                log error $msg
            }
        }
        iq {
            set q {}
            set type  unknown
            set query unknown
            set cmd OnIqQuery
            foreach {a v} $attributes {
                if {$a eq "type"} {
                    set type $v
                } else {
                    lappend q -$a $v
                }
            }
            foreach child $children {
                set elt [wrapper::gettag $child]
                if {$elt eq "query"} {
                    set query [wrapper::getattribute $child xmlns]
                } elseif {$elt eq "vCard"} {
                    set query [wrapper::getattribute $child xmlns]
                    set cmd OnIqVcard
                }
            }
            lappend $q -x $children
            
            if {[catch {eval [linsert $q 0 $cmd $query $type]} errmsg]} {
                log error $errmsg
            }
        }
        default {
            log debug "UNKNOWN: $type $attributes $close $value $children"
        }
    }
    return
}

# ijbridge::OnMessage --
#
#	This is the core procedure. This is called when Jabber messages are
#	received. 'groupchat' messages are from the conference and these we 
#	re-transmit to IRC using the 'xmit' procedure.
#	Some processing occurs here. IRC doesn't accept multi-line posts
#	so we break those up. We also avoid re-sending messages that we
#	have sent from IRC.
#	
#	'chat' messages are messages sent to ijbridge specifically. These
#	are used as commands to the bridge.
# 
proc ::ijbridge::OnMessage {type args} {
    variable Options
    variable lastmsg

    switch -exact -- $type {
        groupchat {
            # xmit to irc

            array set a {-body {} -from {} -x {} -to {}}
            array set a $args
            set jid $Options(Name)@$Options(JID)/$Options(Resource)
            set tobridge [string equal $jid $a(-to)]
            
            # We only deal with groupchat messages sent to the bot except
            # for history messages. We can ignore all other copies as we 
            # are going to place the message on the irc channel.
            # DONT SEND HISTORY FOR NOW.
            if {!$tobridge} {
                return
                if {[info exists a(-x)]} {
                    set nick [jid node $a(-to)]
                    foreach chunk $a(-x) {
                        if {[lsearch -exact [wrapper::getattrlist $chunk] \
                                 jabber:x:delay] != -1} {
                            # We don't xmit history items.
                            foreach line [split $a(-body) \n] {
                                xmit "PRIVMSG $nick :$line"
                            }
                            return
                        }
                    }
                }
                return
            }

            # messages to the bot - don't send irc messages back.
            set stamp [wrapper::getchildwithtaginnamespace \
                           [list {} {} 0 {} $a(-x)] x urn:bridge:timestamp]
            if {$stamp ne ""} { return }

            set nick [jid resource $a(-from)]

            # Messages from the room -- entered/left are done by presence.
            if {$nick eq ""} {
                log debug "skipping room msg \"$a(-body)\""
                return
            }

            log debug "groupchat $args"

            set emote [string match /me* $a(-body)]
            if {$emote} {
                set a(-body) [string range $a(-body) 4 end]
            }
            foreach line [split $a(-body) \n] {
                if {$emote} {
                    xmit "PRIVMSG $::client::channel\
                       :\001ACTION $nick $line\001"
                } else {
                    xmit "PRIVMSG $::client::channel :<$nick> $line"
                }
            }

        }
        normal -
        chat {
            log debug "chat --> $args"
            array set a {-body {} -from {}}
            array set a $args
            # Message to IRC user.
            set from [jid resource $a(-from)]
            set to [jid node $a(-to)]
            if {[string match /me* $a(-body)]} {
                xmit "PRIVMSG $to :\001ACTION $from\
                    [string range $a(-body) 4 end]\001"
            } else {
                xmit "PRIVMSG $to :from $from: $a(-body)"
            }
            return
        }
        error {
            log debug "error --> $args"
            array set a {-body {} -from {}}
            array set a $args
            set to [jid node $a(-to)]
            set code {}
            set err "error recieved"
            foreach child $a(-x) {
                if {[wrapper::gettag $child] eq "error"} {
                    set code [wrapper::getattribute $child code]
                    set err [wrapper::getcdata $child]
                    break
                }
            }
            xmit "PRIVMSG $to :ERROR $code: $err"
        }
        default {
            log info "msg: $type $args"
        }
    }
}

# ijbridge::OnPresence --
#
#	Called when we recieve a Jabber presence notification for any of
#	our irc people.
#
proc ::ijbridge::OnPresence {type args} {
    variable Options
    variable IrcUserList

    log debug "presence: $type $args"
    array set opts {-to {} -from {} -x {} -show {} -status {}}
    array set opts $args

    set jid $Options(Name)@$Options(JID)/$Options(Resource)
    set tobridge [string equal $jid $opts(-to)]
    set nick [jid resource $opts(-from)]

    set stamp [wrapper::getchildwithtaginnamespace \
        [list {} {} 0 {} $opts(-x)] x urn:bridge:timestamp]
    set seen [expr {[llength $stamp] > 0}]

    set msg ""
    if {[string length $opts(-show)] > 0} {
        append msg $opts(-show)
    }
    if {[string length $opts(-status)] > 0} {
        append msg " ($opts(-status))"
    }

    switch -exact -- $type {
        available {
            if {$tobridge} {
                if {$seen} {
                    log notice "skipping presence message for bridged user"
                } else {
                    xmit "PRIVMSG $::client::channel\
                        :\001ACTION $nick is feeling chatty.\001"
                }
            }
        }
        unavailable {
            if {$tobridge && !$seen} {
                xmit "PRIVMSG $::client::channel\
                    :\001ACTION $nick has left.\001"
            }
        }
        probe {
            # If sent to a Irc user and that user not in the IrcUserList
            # then remove the Irc user presence.
            if {[jid domain $opts(-to)] eq $Options(JID)} {
                set nick [jid node $opts(-to)]
                log debug "Received presence probe for '$nick'"
                if {$nick ne $Options(Name)} {
                    set ndx [lsearch $IrcUserList $nick]
                    if {$ndx == -1} {
                        presence $nick unavailable
                    }
                }
            }
        }
        error {
            #for {set n 0} {info exists IrcUserMap(${nick}${n})} {incr n} {
            #    set newnick ${nick}${n}
            #    set IrcUserMap($newnick) $nick
            #    presence $newnick available $opts(-show) $opts(-status)
            #}
        }
        subscribe {
            log notice "presence subscribe from $opts(-from) to $opts(-to)"
        }
        unsubscribe {
            log notice "presence unsubscribe from $opts(-from) to $opts(-to)"
        }
        default {
            log notice "unknown presence type \"$type\""
        }
    }
}

# ijbridge::OnIqQuery --
#
#	Called when we recieve a Jabber info query.
#
proc ::ijbridge::OnIqQuery {query type args} {
    variable Options
    variable Component

    array set a $args
    set jid      $Options(Name)@$Options(JID)/$Options(Resource)
    set tobridge [string equal $jid $a(-to)]
    set nick     [jid resource $a(-from)]
    set rsp      {}
    set qr       {}
    set parts    {}
    switch -exact -- $query {
        jabber:iq:time {
            set time [clock seconds]
            set utc [clock format $time -format %Y%m%dT%H:%M:%S -gmt 1]
            set lct [clock format $time]
            set tz  [clock format $time -format %Z]
            lappend parts [list utc {} 0 $utc {}]
            lappend parts [list tz {} 0 $tz {}]
            lappend parts [list display {} 0 $lct {}]
            lappend qr [list query [list xmlns $query] 0 {} $parts]
            set ra [list xmlns jabber:client type result id $a(-id) \
                        to $a(-from) from $a(-to)]
            set rsp [list iq $ra 0 {} $qr]
        }
        jabber:iq:version {
            lappend parts [list name {} 0 "IRC-Jabber Bridge" {}]
            lappend parts [list version {} 0 $::ijbridge::version {}]
            lappend parts [list os {} 0 "$::tcl_platform(os)\
                               $::tcl_platform(osVersion)" {}]
            lappend qr [list query [list xmlns $query] 0 {} $parts]
            set ra [list xmlns jabber:client type result id $a(-id) \
                        to $a(-from) from $a(-to)]
            set rsp [list iq $ra 0 {} $qr]
        }
        default {
            log debug "iq: $query $type $args"
        }
    }
    if {$rsp != {}} {
        xmppd::jcp::route $Component [wrapper::createxml $rsp]
    }
    return
}

# ijbridge::OnIqVcard --
#
#       Called when we recieve a Jabber vcard query.
#
proc ::ijbridge::OnIqVcard {query type args} {
    variable Options
    variable Component

    array set a {-from {} -to {} -x {} -id 0}
    array set a $args
    set nick     [jid node $a(-to)]

    if {$type ne "get"} {
        log notice "vcard set requested"
    }

    switch -exact -- $query {
        vcard-temp {
            lappend d [list FN {} 0 "IRC user" {}]
            lappend d [list NICKNAME {} 0 $nick {}]
            lappend d [list DESC {} 0 "This user is using IRC" {}]
            set vcard [list vCard {xmlns vcard-temp} 0 "" $d]
            set xattr [list from $a(-to) to $a(-from) type result id $a(-id)]
            set iq [list iq $xattr 0 "" [list $vcard]]
            xmppd::jcp::route $Component [wrapper::createxml $iq]
        }
        default {
            log notice "unknown iq $query $type $args"
        }
    }
} 

# ijbridge::presence --
#
#	Send a jabber presence message for a irc nick.
#
proc ::ijbridge::presence {nick type {show {online}} {status {}} {user {}}} {
    variable Options
    variable Component

    if {$user eq {}} {set user $nick}
    set kids {} ; set hist {}
    set ts [clock format [clock seconds] -format %Y%m%dT%H:%M:%S -gmt 1]
    lappend hist [list history [list maxchars 0 maxstanzas 0] 1 "" {}]
    lappend kids [list x {xmlns http://jabber.org/protocols/muc} 0 "" $hist]
    lappend kids [list x {xmlns urn:bridge:timestamp} 0 $ts {}]
    if {$show ne {}} {
        lappend kids [list show {} 0 $show {}]
    }
    if {$status ne {}} {
        lappend kids [list status {
            xmlns:xml http://www.w3.org/XML/1998/namespace
            xml:lang en-GB
        } 0 $status {}]
    }
    set to $Options(Conference)
    if {$nick ne ""} {append to "/$nick"}
    set from ${user}@$Options(JID)/$Options(Resource)
    set attr [list from $from to $to xmlns jabber:client]
    if {$type ne {}} {lappend attr type $type}
    
    xmppd::jcp::route $Component \
        [wrapper::createxml [list presence $attr 0 "" $kids]]
    return
}

# ijbridge::say --
#
#	Send a jabber message.
#
proc ::ijbridge::say {nick user msg emote} {
    variable Options
    variable Component

    set kids {}
    if {$emote} {set msg "/me $msg"}
    set ts [clock format [clock seconds] -format %Y%m%dT%H:%M:%S -gmt 1]
    #lappend kids [list body {} 0 [wrapper::xmlcrypt $msg] {}]
    lappend kids [list body {} 0 [quote $msg] {}]
    lappend kids [list x {xmlns urn:bridge:timestamp} 0 $ts {}]
    set from "${nick}@$Options(JID)/$Options(Resource)"
    set to   $Options(Conference)
    set type groupchat
    if {$user ne ""} {
        append to "/$user" 
        set type chat
    }
    set attr [list id [mid] from $from to $to type $type xmlns "jabber:client"]
    set xml [wrapper::createxml [list message $attr 0 "" $kids]]
    xmppd::jcp::route $Component $xml
    return
}

# ijbridge::escape --
#
#	Our own version of the jabberlib xmlcrypt that also encodes all the
#	low ascii that is invalid in XML streams.
#
proc ::ijbridge::escape {chdata} {
    variable xmlmap
    if {![info exists xmlmap]} {
        #set xmlmap {& "&amp;" < "&lt;" > "&gt;" "\"" "&quot;" "\'" "&apos;"}
        set xmlmap {}
        for {set n 0} {$n < 32} {incr n} {
            if {$n == 9 || $n == 10 || $n == 13} continue
            lappend xmlmap [format %c $n] [format "&#x%x;" $n]
        }
    }
    return [string map $xmlmap $chdata]
}

proc ::ijbridge::quote {chdata} {
    variable xmlmap
    if {![info exists xmlmap]} {
        set xmlmap {}
        for {set n 0} {$n < 32} {incr n} {
            if {$n == 9 || $n == 10 || $n == 13} continue
            lappend xmlmap [format %c $n] [format "\\%03o" $n]
        }
    }
    return [string map $xmlmap $chdata]
}

# ijbridge::jid --
#
#	A helper function for splitting out parts of Jabber IDs.
#
proc ::ijbridge::jid {part jid} {
    set r {}
    regexp {^(?:([^@]*)@)?([^/]+)(?:/(.+))?} $jid -> node domain resource
    switch -exact -- $part {
        node     { set r $node }
        domain   { set r $domain }
        resource { set r $resource }
        jid      { set r $jid }
        default {
            return -code error "invalid part \"$part\":\
                must be one of node, domain, resource or jid."
        }
    }
    return $r
}    

# ijbridge::xmit --
#
#	This is where we send messages to the IRC channel. IRC requires 
#	rate limiting. A client can be kicked for sending too much in
#	too short a period so here we deal with this.
#
proc ::ijbridge::xmit {str} {
    variable Options
    variable Limit
    # if we are over the sending limit just push line into the queue
    if { $Limit(queue) != "" } {
        lappend Limit(queue) $str
        return
    }
    # count the number of lines per second
    if { ([clock seconds] - $Limit(last)) < 1 } {
        incr Limit(lines)
    } else {
        # remember the last time we sent a line
        set Limit(last) [clock seconds]
        set Limit(lines) 0
    }
    # if we are over the line limit kick off the queued sends
    if { $Limit(lines) > 4 } {
        log info "flood started"
        lappend Limit(queue) $str
        after 1000 [namespace origin XmitFromQueue]
        return
    }
    $client::cn send $str
}

# ijbridge::XmitFromQueue --
#
#	If we had to limit messages sent from 'xmit' then we handle the
#	queued messages from here on a timer.
#
proc ::ijbridge::XmitFromQueue {} {
    variable Options
    variable Limit
    # return if the queue is empty
    if { [string length [set str [lindex $Limit(queue) 0]]] < 1 } {
        set Limit(last) 0
        log info "flood ended"
        return
    }
    set Limit(queue) [lreplace $Limit(queue) 0 0]
    log debug "sending from queue"
    $client::cn send $str
    # send next line
    after 1000 [namespace origin XmitFromQueue]
}

# ijbridge::Pop --
#
#	Utility function used in option processing.
#
proc ::ijbridge::Pop {varname {nth 0}} {
    upvar $varname args
    set r [lindex $args $nth]
    set args [lreplace $args $nth $nth]
    return $r
}

# ijbridge::LoadConfig --
#
#	This procedure reads a text file and updates the Options array
#	from the contents. Comments and blank lines are ignored. All 
#	other lines must be a list of two elements, the first element 
#	must be an item in the Options array.
#
proc ::ijbridge::LoadConfig {} {
    variable Options
    set conf [file normalize [info script]]
    set base [file rootname [file tail $conf]].conf
    set conf [file join [file dirname $conf] $base]
    if {[file exists $conf]} {
        set f [open $conf r]
        set n 0
        while {![eof $f]} {
            gets $f line
            string trim $line
            if {[string match "#*" $line]} continue
            if {[string length $line] < 1} continue
            if {[llength $line] != 2} {
                return -code error "invalid config line $n: \"$line\""
            }
            if {![info exists Options([lindex $line 0])]} {
                return -code error "invalid config option\
                \"[lindex $line 0]\" at line $n"
            }
            set Options([lindex $line 0]) [lindex $line 1]
            incr n
        }
        close $f
    } else {
        log warn "no configuration file found!"
    }
    return
}

# -------------------------------------------------------------------------

# ijbridge::chimes --
#
#	Manage the scheduling of chimes on the hour.
#
proc ::ijbridge::chimes {cmd} {
    variable ChimeId
    switch -exact -- $cmd {
        start {
            set ChimeId [after [nextchime] [namespace origin bong]]
        }
        stop {
            after cancel $ChimeId
        }
        default {
            return -code error "invalid option \"$cmd\": rtfm"
        }
    }
}

# ijbridge::nextchime --
#
#	Calculate the number of milliseconds until the next hour.
#
proc ::ijbridge::nextchime {} {
    set t [clock format [clock scan "+1 hour"] -format "%Y%m%d %H:00:00"]
    set delta [expr {([clock scan $t] - [clock seconds]) * 1000}]
    if {$delta < 60000} {
        log error "error: chiming too fast"
        set delta 60000
    }
    log debug "Schedule chime in $delta milliseconds"
    return $delta
}

# ijbridge::bong --
#
#	Issue a timestamp message to the connected chatroom.
#
proc ::ijbridge::bong {} {
    variable ChimeId
    variable Options
    variable Component

    after cancel $ChimeId
    set kids {}
    set ts [clock format [clock seconds] -format %Y%m%dT%H:%M:%S -gmt 1]
    puts "BONG at $ts"
    lappend kids [list body {} 0 \
                      [clock format [clock seconds] -gmt 1] {}]
    set from "$Options(Name)@$Options(JID)/$Options(Resource)"
    set attr [list from $from to $Options(Conference) \
                  type groupchat xmlns "jabber:client"]
    set xml [wrapper::createxml [list message $attr 0 "" $kids]]

    xmppd::jcp::route $Component $xml
    set ChimeId [after [nextchime] [namespace origin bong]]
}

# -------------------------------------------------------------------------

# create a server connection and set up associated events

namespace eval ::client {
    variable channel
    if {![info exists channel]} {set channel ""}

    variable log
    if {![info exists log]} {
        set log [logger::init irc]
        ${log}::setlevel $::ijbridge::Options(LogLevel)
        proc ${log}::stdoutcmd {level text} {
            variable service
            set ts [clock format [clock seconds] -format {%H:%M:%S}]
            puts "\[$ts\] $level $text"
        }
        proc log {level msg} {
            variable log
            ${log}::${level} $msg
        }
    }

}

proc ::client::start {} {
    variable ::ijbridge::Options
    create $Options(IrcServer) $Options(IrcPort) \
        $Options(IrcUser) $Options(IrcChannel)
}

proc ::client::stop {} {
    if {[info exists cn]} {
        $cn quit
    }
}

proc ::client::create { server port nk chan } {
    variable ::ijbridge::Options
    variable cn
    variable channel $chan
    variable nick $nk
    set cn [::irc::connection]

    $cn registerevent 001 {
        ::client::log debug "event 001 [target]"
        set ::client::nick [target]
        cmd-join $::client::channel
    }

    $cn registerevent 433 {
        if { [lindex [additional] 0] == $::client::nick } {
            cmd-send "NICK [string trimright $::client::nick 0123456789][string range [expr rand()] end-2 end]"
        }
    }

    $cn registerevent defaultcmd {
	::client::log debug "[action]:[msg]"
    }

    $cn registerevent defaultnumeric {
	::client::log debug "[action]:[target]:[msg]"
    }

    $cn registerevent defaultevent {
        if {[action] ne "PONG"} {
            ::client::log debug "[action]:[who]:[target]:[msg]"
        }
    }

    $cn registerevent 353     {::client::userlist [who] [target] [msg] }
    $cn registerevent PART    {::client::part     [who] [target] [msg] }
    $cn registerevent JOIN    {::client::join     [who] [target] [msg] }
    $cn registerevent NICK    {::client::nick     [who]          [msg] }
    $cn registerevent PRIVMSG {::client::privmsg  [who] [target] [msg] }
    $cn registerevent NOTICE  {::client::privmsg  [who] [target] [msg] }
    $cn registerevent QUIT    {::client::quit     [who] [target] [msg] }

    $cn registerevent EOF "
        ::client::log notice \"Disconnected from IRC\"
        ::client::connect \$::client::cn $server $port
    "

    connect $cn $server $port
}

# -------------------------------------------------------------------------

proc ::client::join {who target msg} {
    variable ::client::::nick
    variable ::ijbridge::IrcUserList

    if { $who != $nick } {
        ijbridge::presence $who available
        
        if { [lsearch $IrcUserList $who] == -1 } {
            lappend IrcUserList $who
        }
    }    
}

proc ::client::part {who target msg} {
    variable ::client::nick
    variable ::client::channel
    variable ::ijbridge::IrcUserList

    if { $target == $channel && $who != $nick } {
        ijbridge::presence $who unavailable
        set item [lsearch $IrcUserList $who]
        if { $item > -1 } {
            set IrcUserList [lreplace $IrcUserList $item $item]
        }
    }
}

proc ::client::nick {old new} {
    variable ::client::nick
    variable ::ijbridge::Options
    variable ::ijbridge::IrcUserList

    log debug "NICK $old $new"
    if { $old eq $nick } {
        set nick $new
    }

    ijbridge::say $old {} "changed their name to $new" 1
    #ijbridge::presence $new {} {} {} $old;# leaves them as olduser@domain
    ijbridge::presence $old unavailable
    ijbridge::presence $new available
    set ndx [lsearch $IrcUserList $old]
    if { $ndx > -1 } {
        set IrcUserList [lreplace $IrcUserList $ndx $ndx]
    }
    lappend IrcUserList $new
    ijbridge::presence {} probe {} {} $Options(Name)
    return
}

proc ::client::quit {who target msg} {
    variable ::client::nick
    log notice "QUIT $who $target $msg"
    if { $who != $nick } {
        ijbridge::presence $who unavailable
        if { $who eq [string trimright $nick 0123456789] } {
            cmd-send "NICK $who"
        }
    }
}

#List of online users sent on channel join
proc ::client::userlist {who target msg} {
    variable ::ijbridge::IrcUserList
    variable ::ijbridge::Options
    variable ::client::nick
    
    log notice "UsersOnline '$who' '$target' $msg"
    set IrcUserList \
        [split [string map {@ "" % "" + ""} [string trim $msg]] " "]
    foreach bridge [list ircbridge azbridge ijbridge $nick] {
        set item [lsearch $IrcUserList $bridge]
        if { $item > -1 } {
            set IrcUserList [lreplace $IrcUserList $item $item]
        }
    }
    # Send a probe presence to the bridge, this results in 
    # probes being sent to each current IRC user. We can intercept
    # these and match against the IrcUserList and remove those no-longer
    # present
    ijbridge::presence {} probe {} {} $Options(Name)

    # Now lets add the missing nicks.
    foreach nick $IrcUserList {
        ijbridge::presence $nick available online
    }
    return
}

proc ::client::privmsg {who target msg} {    
    variable ::client::nick
    variable ::client::channel
    log debug "privmsg:$who:$target:$msg"

    if {$who eq "AUTH" || $who eq ""} {
        return
    }

    set emote 0
    if {[regexp {^\001(\w+) ?(.+)?\001$} $msg -> cmd msg]} {
        switch -exact -- $cmd {
            ACTION { set emote 1 }
            default {
                log info "$who CTCP $cmd $msg"
                return
            }
        }
    }
    if { $target eq $channel } {
        ijbridge::say $who {} $msg $emote
    } elseif { $target eq $nick } {
        set split [split $msg " \t"]
        set redir [lindex $split 0]
        set emote [string equal "/me" [lindex $split 1]]
        set msg [string range $msg [string length $redir] end]
        if {$emote} {set msg [string range $msg 5 end]}
        ijbridge::say $who $redir $msg $emote
    } else {
        ijbridge::say $who $target $msg $emote
    }
}

# -------------------------------------------------------------------------

# connect to the server and register

proc ::client::connect {cn server port} {
    variable ::ijbridge::Limit
    variable nick
    # set up variable for rate limiting
    array set Limit [list last [clock seconds] queue {} lines 0]
    log notice "Connecting to $server on port $port"
    if {[catch {$cn connect $server $port} err]} {
        log notice "Could not connect: $err"
        after 10000 [list [namespace current]::connect $cn $server $port]
        return
    }
    $cn user $nick localhost domain \
        "Tcl Jabber-IRC bridge v$::ijbridge::version"
    $cn nick $nick
    ping
    log notice "Connected to $server"
}

proc ::client::ping {} {
    variable cn
    $cn serverping
    after cancel [namespace current]::ping
    after 20000 [namespace current]::ping
}

proc bgerror {args} {
    global errorInfo
    ::client::log error "BGERROR: [join $args]"
    ::client::log error "ERRORINFO: $errorInfo"
}

::irc::config debug 0

# -------------------------------------------------------------------------

::ijbridge::LoadConfig
::ijbridge::loglevel $::ijbridge::Options(LogLevel)

if {!$tcl_interactive} {

    # Setup control stream.
    if {$tcl_platform(platform) eq "unix"} {
        set cmdloop [file join [file dirname [info script]] cmdloop.tcl]
        puts "Load $cmdloop"
        if {[file exists $cmdloop]} {
            source $cmdloop
            set cmdloop::welcome "Tcl IRC-Jabber bridge $::ijbridge::version"
            append cmdloop::welcome "\nReady for input from %client %port"
            cmdloop::cmdloop
            cmdloop::listen 0.0.0.0 5441
        }
        set tcl_interactive 1; # fake it so we can re-source this file
    }

    # Begin the bridge
    ::ijbridge::start
    ::client::start

    # Loop forever, dealing with Wish or Tclsh
    if {[info exists tk_version]} {
        if {[tk windowingsystem] eq "win32"} { console show }
        wm withdraw .
        tkwait variable ::forever
        ijbridge::stop
    } else {
        # Permit running as a Windows service.
        if {![info exists tcl_service]} {
            vwait ::forever
            ijbridge::stop
        }
    }
}
